﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.Common.Extensions;
using VA.PPMS.IWS.MappingService.Helpers;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.MappingService.Mappers
{
    public class MapProviderCredentialToCrm : MapperRelatedBase
    {
        public MapProviderCredentialToCrm(IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper)
            : base(ppmsContextHelper, ppmsHelper)
        {}

        public async Task<ppms_providercredential> MapInsert(ProviderCredential credential, Account provider)
        {
            // map entity
            var entity = new ppms_providercredential
            {
                Id = Guid.NewGuid(),
                ppms_name = credential.Name,
                ppms_Description = credential.Description,
                ppms_credentialingstatus = EnumHelper.MapEnumToOptionSetValue<ppms_providercredential_ppms_credentialingstatus>(credential.CredentialingStatus.ToString()),
                ppms_credentialtype = EnumHelper.MapEnumToOptionSetValue<ppms_providercredential_ppms_credentialtype>(credential.CredentialType)
            };


            if (credential.CredentialedDate.Year > EnumHelper.MinYear) entity.ppms_credentialeddate = credential.CredentialedDate.ToCrmDate();

            if (credential.NextCredentialingDateSpecified && credential.NextCredentialingDate.Year > EnumHelper.MinYear) entity.ppms_nextcredentialingdate = credential.NextCredentialingDate.ToCrmDate();

            // Set owner to CCN
            if (!ForVaNetwork && Owner != null) entity.OwnerId = Owner;

            await Task.Run(() => {});

            return entity;
        }

        public async Task<ppms_providercredential> MapUpdate(ProviderCredential entity, Account provider)
        {
            // get entity
            var crmEntity = GetCrmEntity(provider, entity.Name);

            // matching account not found
            if (crmEntity == null)
            {
                return await MapInsert(entity, provider);
            }

            var newEntity = new ppms_providercredential()
            {
                Id = crmEntity.Id,
                ppms_name = crmEntity.ppms_name,
                ppms_Description = crmEntity.ppms_Description,
                ppms_credentialeddate = crmEntity.ppms_credentialeddate,
                ppms_credentialingstatus = crmEntity.ppms_credentialingstatus,
                ppms_credentialtype = crmEntity.ppms_credentialtype,
                ppms_nextcredentialingdate = crmEntity.ppms_nextcredentialingdate
            };

            // Map fields
            if (IsChanged(entity.Description, newEntity.ppms_Description)) newEntity.ppms_Description = entity.Description;

            newEntity.ppms_credentialingstatus = EnumHelper.MapEnumToOptionSetValue<ppms_providercredential_ppms_credentialingstatus>(entity.CredentialingStatus.ToString());
            newEntity.ppms_credentialtype = EnumHelper.MapEnumToOptionSetValue<ppms_providercredential_ppms_credentialtype>(entity.CredentialType);

            if (entity.CredentialedDate.Year > EnumHelper.MinYear) newEntity.ppms_credentialeddate = entity.CredentialedDate.ToCrmDate();

            if (entity.NextCredentialingDateSpecified && entity.NextCredentialingDate.Year > EnumHelper.MinYear) newEntity.ppms_nextcredentialingdate = entity.NextCredentialingDate.ToCrmDate();

            // Set owner to CCN
            if (!ForVaNetwork && Owner != null) newEntity.OwnerId = Owner;

            return newEntity;
        }

        private static ppms_providercredential GetCrmEntity(Account provider, string id)
        {
            var list = provider.ppms_account_ppms_providercredential;
            if (list == null) return null;

            var ppmsProvidercredentials = list as ppms_providercredential[] ?? list.ToArray();

            return ppmsProvidercredentials.Any() ? ppmsProvidercredentials.FirstOrDefault(x => x.ppms_name == id) : null;
        }

        private static ProviderCredential ConvertEntity<T>(T entity)
        {
            return (ProviderCredential)Convert.ChangeType(entity, typeof(ProviderCredential));
        }

        public override async Task<Entity> MapUpdate<T>(T entity, Entity parent)
        {
            return await MapUpdate(ConvertEntity(entity), (Account)parent);
        }

        public override async Task<Entity> MapInsert<T>(T entity, Entity parent)
        {
            return await MapInsert(ConvertEntity(entity), (Account)parent);
        }

        public override void AddChildrenToProvider(IList<Entity> entities, Entity parent)
        {
            if (IsWithinContext) AssociateRelatedEntities(parent, entities, "ppms_account_ppms_providercredential");
            else
            {
                if (entities != null && entities.Count > 0)
                {
                    var account = (Account)parent;
                    if (account != null) account.ppms_account_ppms_providercredential = ConvertEntityList<ppms_providercredential>(entities);
                }
            }
        }

        public override IEnumerable<SetStateRequest> MapDelete<T>(IList<T> entities, Entity parent)
        {
            if (entities == null || !entities.Any()) return null;

            // Check provider
            var provider = (Account)parent;
            if (provider == null) return null;

            var list = new List<ppms_providercredential>();

            // Map schema entities for delete
            foreach (var item in entities)
            {
                var entity = ConvertEntity(item);
                var matches = provider.ppms_account_ppms_providercredential.Where(p => p.ppms_name == entity.Name);
                list.AddRange(matches);
            }

            return EntityDelete((IEnumerable<ppms_providercredential>)list);
        }
    }
}